SUBROUTINE plot ( y, npts, minplt, maxplt, default, unit )
!
!  Purpose:
!    Subroutine to plot the points in array y.  The data in the
!    array is assumed to be at a uniform spacing. 
!
!  Record of revisions:
!      Date       Programmer          Description of change
!      ====       ==========          =====================
!    11/27/95    S. J. Chapman        Original code
!
IMPLICIT NONE

! Declare calling arguments:
INTEGER, INTENT(IN) :: npts      ! Number of points to plot
REAL,DIMENSION(npts),INTENT(IN) :: y  ! Input data
REAL, INTENT(IN) :: minplt       ! Minimum value of plot
REAL, INTENT(IN) :: maxplt       ! Maximum value of plot
LOGICAL, INTENT(IN) :: default   ! Flag to set default limits
INTEGER, INTENT(IN) :: unit      ! Output i/o unit to plot on

! External functions:
CHARACTER(len=12) :: real_to_char  ! Convert real to char str

! Declare parameters:
INTEGER, PARAMETER :: nbins = 65   ! Number of bins to plot in
 
! Declare local variables:
CHARACTER(len=14) :: annotation  ! Line annotation (y value)
CHARACTER(len=12) :: ch_maxamp   ! Char form of max value to plot
CHARACTER(len=12) :: ch_minamp   ! Char form of min value to plot
INTEGER :: i                     ! Loop index
INTEGER :: ibin                  ! Bin # for current y value
INTEGER :: ibin0                 ! Bin # for zero crossing
REAL :: fraction                 ! Fraction of plot width 
REAL :: maxamp                   ! Max value to plot (local)
REAL :: minamp                   ! Min value to plot (local)
CHARACTER(len=65) :: plot_buffer ! Plotting buffer
CHARACTER(len=65) :: scale       ! Scale on border of plot

! If the scales are defaulted, set min and max of Y axis. 
set_range: IF ( default ) THEN
   ! Get the largest and smallest values in the array.
   maxamp = MAXVAL(y)
   minamp = MINVAL(y)
ELSE 
   ! Set specified values for range of Y axis.
   maxamp = maxplt
   minamp = minplt
END IF set_range
 
! We will divide minamp to maxamp into 65 bins for plotting
! purposes.  Locate the zero bin, if it is between minamp 
! and maxamp.
IF ( (maxamp > 0.) .AND. (minamp < 0) ) THEN
   fraction = ( 0. - minamp) / (maxamp - minamp )
   ibin0 = NINT ( (nbins-1) * fraction ) + 1
ELSE
   ibin0 = 0
END IF
 
! Set border scale, including zero mark.
annotation = ' '
scale = '+-------------------------------&
        &--------------------------------+'
IF ( ibin0 > 0 ) THEN
   scale(ibin0:ibin0) = '+'
END IF
 
! Print upper border.
ch_minamp = real_to_char(minamp)
ch_maxamp = real_to_char(maxamp)
WRITE (unit,'(10X,A,46X,A)') ch_minamp, ch_maxamp
WRITE (unit,'(A,1X,A)') annotation, scale
 
! Plot data points.  
plot_points: DO i = 1, npts
 
   ! Clear line
   plot_buffer = ' '       
   annotation  = ' ' 
 
   ! Set value of y data point.
   annotation(2:13) = real_to_char( y(i) )
 
   ! Set min and max borders.
   plot_buffer(1:1)   = '|'
   plot_buffer(65:65) = '|'
 
   ! Set zero line, if within borders.
   IF ( ibin0 > 0 ) THEN
      plot_buffer(ibin0:ibin0) = '|'
   END IF
 
   ! Plot point on array.
   fraction = ( y(i) - minamp) / (maxamp - minamp )
   ibin = NINT ( (nbins-1) * fraction ) + 1
   IF ( (ibin >= 1) .AND. (ibin <= nbins) ) THEN
      plot_buffer(ibin:ibin) = '*'
   END IF
 
   ! Write out line.
   WRITE (unit,'(A,1X,A)') annotation, plot_buffer

END DO plot_points
 
! Print lower border.
annotation = ' '
WRITE (unit,'(A,1X,A)') annotation, scale
WRITE (unit,'(10X,A,46X,A)') ch_minamp, ch_maxamp
 
! Print out summary info.
WRITE (unit,'(/,10X,A,I12)' ) 'Number of Points = ', npts
 
END SUBROUTINE plot
